<?php

namespace App\Http\Controllers;

use App\Models\Folder;
use App\Models\User;
use Exception;
use Illuminate\Http\Request;
use SergiX44\Nutgram\Nutgram;
use SergiX44\Nutgram\Telegram\Exceptions\TelegramException;
use Throwable;

class TelegramBroadcastController extends Controller
{
    public function index()
    {
        $folders = Folder::on()->whereNull('parent_id')->orderBy('name')->get();
        return view('admin.broadcast.index', compact('folders'));
    }

    public function broadcast(Request $request, Nutgram $bot)
    {
        $request->validate([
            'message' => 'required|string|max:4000', // Adjust max length as needed
            'audience' => 'required|in:all,folder,specific',
            'folder_id' => 'required_if:audience,folder|exists:folders,id',
            'specific_id' => 'required_if:audience,specific|integer|min:1',
        ]);

        $message_text = $request->input('message');
        $audience = $request->input('audience');
        $specific_id = $request->input('specific_id');

        $results = [];
        $chats = [];

        try {
            if ($audience === 'all') {
                $chats = User::whereNotNull('telegram_id')->pluck('telegram_id')->toArray();
                $results[] = '<span class="text-blue-500">Targeting all users.</span>';
            } elseif ($audience === 'folder') {
                $folder_id = $request->input('folder_id');
                $folder = Folder::with('users')->findOrFail($folder_id);
                $chats = $folder->users()->whereNotNull('telegram_id')->pluck('telegram_id')->toArray();
                $results[] = '<span class="text-blue-500">Targeting users of folder: ' . htmlspecialchars($folder->name) . '</span>';
            } elseif ($audience === 'specific') {
                if (empty(trim($specific_id))) {
                    throw new Exception("A valid, numeric Telegram ID must be provided.");
                }
                $chats = [$specific_id]; // Wrap in array
                $results[] = '<span class="text-blue-500">Targeting specific ID: ' . htmlspecialchars($specific_id) . '</span>';
            }

            if (empty($chats)) {
                $results[] = '<span class="text-yellow-500 font-bold">Warning: No users found for the selected audience.</span>';
            } else {
                $success_count = 0;
                $error_count = 0;

                foreach ($chats as $chat_id) {
                    try {
                        $bot->sendMessage(
                            text: $message_text,
                            chat_id: $chat_id,
                            parse_mode: 'html'
                        );
                        $success_count++;
                    } catch (TelegramException $th) {
                        $error_count++;
                        $results[] = "❌ ERROR for ChatID: " . htmlspecialchars($chat_id) . " | " . htmlspecialchars($th->getMessage());
                    }
                }

                $results[] = "<hr class=\"my-2 border-gray-600\"><span class=\"font-bold\">Broadcast Complete.</span>";
                $results[] = "<span class=\"text-green-500\">Successful sends: $success_count</span>";
                $results[] = "<span class=\"text-red-500\">Failed sends: $error_count</span>";
            }
        } catch (Throwable $e) {
            $results[] = '<span class="text-red-500 font-bold">A critical error occurred: ' . htmlspecialchars($e->getMessage()) . '</span>';
        }

        $folders = Folder::on()->whereNull('parent_id')->orderBy('name')->get();
        return view('admin.broadcast.index', [
            'folders' => $folders,
            'message_text' => $message_text,
            'audience' => $audience,
            'specific_id_value' => $specific_id,
            'results' => $results,
        ]);
    }
}
