<?php

namespace App\Telegram\Handlers;

use App\Models\Asset;
use App\Models\Folder;
use App\Models\User;
use App\Telegram\EntityDecoder;
use App\Telegram\Menus\Admin\Helpers;
use App\Telegram\Menus\InlineMenuPaginator;
use Illuminate\Pagination\LengthAwarePaginator;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use SergiX44\Nutgram\Nutgram;
use SergiX44\Nutgram\Telegram\Properties\MessageType;
use SergiX44\Nutgram\Telegram\Properties\ParseMode;
use SergiX44\Nutgram\Telegram\Types\Keyboard\InlineKeyboardButton;

class MainHandler extends InlineMenuPaginator
{
    use Helpers;

    public function start(Nutgram $bot): void
    {
        $this->handleUserCreation($bot);

        $this->renderPage($bot, null, 1);
    }


    protected function renderPage(Nutgram $bot, int|null $folderId, int $page, array $data = null): void
    {
        $user = User::where('telegram_id', $bot->userId())->with('folders')->first();

        if (!$user || $user->folders->isEmpty()) {
            $this->menuText('You do not have access to any content yet.')->showMenu();
            return;
        }


        $explicitlyAccessibleIds = $user->folders->pluck('id');


        $query = <<<SQL
        WITH RECURSIVE accessible_folders AS (
            SELECT id, parent_id, name FROM folders WHERE id IN (?)

            UNION ALL

            SELECT f.id, f.parent_id, f.name FROM folders f
            JOIN accessible_folders af ON f.parent_id = af.id
        )
        SELECT id FROM accessible_folders
        SQL;


        $bindings = implode(',', array_fill(0, count($explicitlyAccessibleIds), '?'));
        $query = str_replace('IN (?)', "IN ({$bindings})", $query);

        $allAccessibleFolderIds = collect(DB::select($query, $explicitlyAccessibleIds->toArray()))->pluck('id');


        $currentFolder = $folderId ? Folder::find($folderId) : null;


        if ($currentFolder && !$allAccessibleFolderIds->contains($currentFolder->id)) {
            $bot->answerCallbackQuery('Access Denied.');
            $this->renderPage($bot, null, 1); // Go back to root
            return;
        }

        $subfolders = Folder::where('parent_id', $folderId)
            ->whereIn('id', $allAccessibleFolderIds)
            ->orderByRaw('position IS NULL, position ASC')
            ->orderBy('created_at')
            ->get();
        $subFoldersCount = $subfolders->count();

        $assets = Asset::on()->where('folder_id', $folderId)
            ->orderByRaw('position IS NULL, position ASC')
            ->orderBy('created_at')
            ->get();
        $assetsCount = $assets->count();

        $path = $currentFolder ? $currentFolder->getPath() : 'Home';
        $menuText = "📂 **{$path}**";

        if ($currentFolder && $currentFolder->description) {
            $menuText .= "\nℹ️ " . $currentFolder->description;
        }
        $menuText .= "\n\n📚 Items: " . $subFoldersCount . ' folders / ' . $assetsCount . ' assets';
        $this->menuText($menuText, ['parse_mode' => 'Markdown'])->clearButtons();


        $items = new Collection([...$subfolders, ...$assets]);

        $paginator = new LengthAwarePaginator($items->forPage($page, $this->perPage), $items->count(), $this->perPage, $page);

        if ($paginator->isEmpty()) {
            $this->menuText("📂 **{$path}**\n\nThis folder is empty.", ['parse_mode' => 'Markdown']);
        }

        if ($assets->count() > 1) {
            $this->addButtonRow(InlineKeyboardButton::make(
                '💾 Download all content of this folder',
                callback_data: "{$folderId}@handleAllAssets"
            ));
        }

        foreach ($paginator->items() as $item) {
            if ($item instanceof Folder) {
                $this->addButtonRow(InlineKeyboardButton::make(
                    "📁 {$item->name}",
                    callback_data: "{$item->id}:{$page}@handleNavigation"
                ));
            } elseif ($item instanceof Asset) {
                $this->addButtonRow(InlineKeyboardButton::make(
                    "{$item->getIcon()} {$item->title}",
                    callback_data: "{$item->id}@handleAsset"
                ));
            }
        }

        $this->addControls($paginator, $currentFolder);

        $this->showMenu();
    }


    function handleAllAssets(Nutgram $bot): void
    {
        $bot->answerCallbackQuery();
        $folderId = (int)$bot->callbackQuery()->data;

        $assets = Asset::on()->where('folder_id', $folderId)
            ->orderByRaw('position IS NULL, position ASC')
            ->orderBy('created_at')
            ->get();

        foreach ($assets as $asset) {
            $this->sendAsset($bot, $asset);
        }

        $this->end();

        $this->renderPage($bot, $folderId, 1);
    }

    public function handleAsset(Nutgram $bot): void
    {
        $bot->answerCallbackQuery();
        $asset = Asset::find((int)$bot->callbackQuery()->data);

        if (!$asset) return;

        $this->sendAsset($bot, $asset);
    }

    private function sendAsset(Nutgram $bot, Asset $asset): void
    {
        $metadata = json_decode(json_encode($asset->metadata), true);
        $caption = 'Asset ID: ' . $asset->id . "\n\n" . $asset->title;
        if ($metadata['caption']) {
            $caption .= "\n" . "<blockquote>" . $metadata['caption'] . "</blockquote>";
        }

        try {
            match ($asset->type) {
                MessageType::PHOTO->name => $bot->sendPhoto($asset->file_id, caption: $caption, parse_mode: ParseMode::HTML),
                MessageType::VIDEO->name => $bot->sendVideo($asset->file_id, caption: $caption, parse_mode: ParseMode::HTML),
                MessageType::DOCUMENT->name => $bot->sendDocument($asset->file_id, caption: $caption, parse_mode: ParseMode::HTML),
                MessageType::AUDIO->name => $bot->sendAudio($asset->file_id, caption: $caption, parse_mode: ParseMode::HTML),
                MessageType::TEXT->name => (function () use ($bot, $asset) {
                    $entity_decoder = new EntityDecoder('HTML');
                    $decoded_text = $entity_decoder->decode(json_decode(json_encode($asset->metadata)));

                    $bot->sendMessage($decoded_text, parse_mode: ParseMode::HTML);
                })(),
                default => $bot->sendMessage("Unsupported asset type."),
            };
        } catch (\Exception $e) {
            $bot->sendMessage("Sorry, there was an error sending the asset." . $e->getMessage());
        }
    }
}
